#!/bin/bash
# bootstrap.sh - Creates an in-tree installation of the mcsh package.
#
# This script enables the majority of development and installation
# features provided by the MCSH development scripts.  Because of this
# circular dependency (wherein MCSH requires MCSH to build), this script
# must be run before any other actions.  Indeed, attempting to run
# ``make`` prior to running this script will produce an error message
# that explains you must run this script.
#
# This script is also useful in situations where bugs have been
# introduced into the in-tree installation during development.  Because
# it starts from scratch, this script can be the simplest means of
# recovering from such a situation.  However, the MCSH Development
# Environment provides other mechanisms for recovering from minor errors.
#
# This script will not change or delete the configuration or data files
# in your local installation tree directly, so it should be safe to be
# called repeatedly without negative consequences.

set -e

declare trace=${trace:-false}
! $trace || set -x

declare verbose=${verbose:-false}


######
# Package Configuration

declare PKG PREFIX
PKG=mcsh
PREFIX=install

declare -a make_opts
! $verbose || make_opts+=( --quiet )


######
# Helper Functions

_find_files() { find "$1" -name '*.in' | sed -e "s,^$1/,," -e 's,\.in$,,'; }
find_files() { _find_files "$@" | xargs echo; }
find_tools() { find_files "src" | sed -e "s,^[^/]*/,,g"; }
find_cats() { find_files "src" | grep -v '/' | sed -e "s,^/*,," | sort -u; }

run() { echo "$@" >&2; "$@"; }
run_sudo() { run sudo -H "$@"; }


######
# Native Dependencies

declare -ga mcsh_deps
mcsh_deps=( )

check_dep() {
	local pkg=$1
	local bin=${2:-$1}
	local path
	path=$(which "$bin") || echo "$pkg: missing dependency" >&2
	[ "$path" ] || mcsh_deps+=( "$pkg" )
}

install_deps() {
	[ "${mcsh_deps[*]}" ] || return 0
	run_sudo apt install --yes "${mcsh_deps[@]}"
}

### CORE IDE DEPENDENCIES ONLY ###
check_dep bc
check_dep coreutils yes
### CORE IDE DEPENDENCIES ONLY ###

install_deps

######
# Package Configuration Stubs

declare tools cats apps libs
cats=$(find_cats)
tools=$(find_tools)
apps=$(find_files 'src')
libs=$(find_files 'libs')

# Create system.i7 and site.i7, as Makefile.i7 depends on them
touch system.i7 site.i7

# Create Makefile.i7
cat >Makefile.i7<<EOF
PKG = $PKG
RUNTIME = $PKG
VERSION = unknown
AUTHOR = unknown
URL = unknown
SUPPORT_EMAIL = unknown
SUPPORT_NAME = unknown
PREFIX = $PREFIX
EXEC_PREFIX = $PREFIX
DATA_PREFIX = $PREFIX
CATS = $cats
APPS = $apps
TOOLS = $tools
ALIBS = $libs
EOF


######
# Bootstrap Setup

# Create system.i7 file
run make dev-config

# avoid rebuilding Makefile.i7 or license again due to new system.i7
run mkdir -p obj/gen/libs/core
run touch Makefile.i7 libs/core/license-parts.in


######
# Boostrap Installation

# Initial install to use for building
run make "${make_opts[@]}" install

# Force Makefile.i7 to regenerate
run rm Makefile.i7

# Rebuild and install everything
run make "${make_opts[@]}" install
